#ifndef LZ_EASING_GLSL
#define LZ_EASING_GLSL
/*
easing.glsl
Implementation of various 1D easing functions, including a generic Cubic and Quintic bezier
*/

#ifndef PI
#define PI 3.1415926535897932384626433832795028841971693993751058209749445923078164062
#define DPI 3.1415926535897932384626433832795028841971693993751058209749445923078164062 * 2.0
#define PIH 3.1415926535897932384626433832795028841971693993751058209749445923078164062 * 0.5
#endif

//Special
float powContrast(float t, float mid, float power) {
	return pow(t / mid, power) * mid;
}

//Cubic bezier
float bezierQuadratic(float t, float a, float b, float c) {
	return a*pow(1 - t, 2) + b * 2 * t*(1 - t) + c*t*t;
}

float bezierCubic(float t, float a, float b, float c, float d) {
	return a*pow(1 - t, 3) + b * 3 * t*pow(1 - t, 2) + c * 3 * pow(t, 2)*(1 - t) + d*pow(t, 3);
}

//Polynmial
//See: http://www.timotheegroleau.com/Flash/experiments/easing_function_generator.htm
float polynomialQuintic(float t, float a, float b, float c, float d, float e) {
	float tc = pow(t, 3);
	float ts = t*t;
	return a*tc*ts + b*ts*ts + c*tc + d*ts + e*t;
}

//Quadratic
float quadraticIn(float t) { return t*t; }
float quadraticOut(float t) { return -t*(t - 2.0f); }
float quadraticInOut(float t) {
	t *= 2.0f;
	if(t < 1.0f)
		return 0.5f*t*t;
	--t;
	return -0.5f*(t*(t - 2) - 1);
}

//Cubic
float cubicIn(float t) { return t*t*t; }
float cubicOut(float t) {
	--t;
	return t*t*t + 1;
}
float cubicInOut(float t) {
	t *= 2.0f;
	if(t < 1.0f)
		return 0.5f*t*t*t;
	t -= 2;
	return 0.5f*(t*t*t + 2);
}
float cubicElasticIn(float t) {
	return (4 * t*t*t - 3 * t*t);
}
float cubicElasticOut(float t) {
	return (4 * t*t*t - 9 * t*t + 6 * t);
}


//Quartic
float quarticIn(float t) { return t*t*t*t; }
float quarticOut(float t) {
	--t;
	return -(t*t*t*t - 1);
}
float quarticInOut(float t) {
	t *= 2.0f;
	if(t < 1.0f)
		return 0.5f*t*t*t*t;
	t -= 2;
	return -0.5f*(t*t*t*t - 2);
}

float quarticElasticIn(float t) {
	return (2 * t*t*t*t + 2 * t*t*t - 3 * t*t);
}
float quarticElasticOut(float t) {
	return (-2 * t*t*t*t + 10 * t*t*t - 15 * t*t + 8 * t);
}

//Quintic
float quinticIn(float t) { return t*t*t*t*t; }
float quinticOut(float t) {
	--t;
	return t*t*t*t*t + 1;
}
float quinticInOut(float t) {
	t *= 2.0f;
	if(t < 1.0f)
		return 0.5f*t*t*t*t*t;
	t -= 2;
	return 0.5f*(t*t*t*t*t + 2);
}

//Sinusoidal
float sinusIn(float t) { return 1.0f - cos(t * PIH); }
float sinusOut(float t) { return sin(t * PIH); }
float sinusInOut(float t) { return 0.5f - 0.5f * cos(t*PI); }

//Exponential
float expIn(float t) { return pow(2.0f, 10.0f*(t - 1)); }
float expOut(float t) { return 1.0f - pow(2.0f, -10.0f*t); }
float expInOut(float t) {
	t *= 2.0f;
	if(t < 1.0f)
		return 0.5f*pow(2.0f, 10.0f*(t - 1));
	--t;
	return 0.5f * (2.0f - pow(2.0f, -10.0f*t));
}

//Circular
float circIn(float t) { return 1 - sqrt(1 - t*t); }
float circOut(float t) {
	--t;
	return sqrt(1 - t*t);
}
float circInOut(float t) {
	t *= 2.0f;
	if(t < 1.0f)
		return 0.5f*(1 - sqrt(1 - t*t));
	t -= 2.0f;
	return 0.5f * (1 + sqrt(1 - t*t));
}

//Elastic
float elasticSoftOut(float t) {
	float ts = t*t;
	float tc = t*ts;
	return (33 * tc*ts - 106 * ts*ts + 126 * tc - 67 * ts + 15 * t);
}

float elasticSoftIn(float t) {
	float ts = t*t;
	float tc = t*ts;
	return (33 * tc*ts - 59 * ts*ts + 32 * tc - 5 * ts);
}

float elasticHardOut(float t) {
	float ts = t*t;
	float tc = t*ts;
	return (56 * tc*ts - 175 * ts*ts + 200 * tc - 100 * ts + 20 * t);
}

float elasticHardIn(float t) {
	float ts = t*t;
	float tc = t*ts;
	return (56 * tc*ts - 105 * ts*ts + 60 * tc - 10 * ts);;
}

/*
* easing.glsl end
*/
#endif